            #include "HWT.h"
#include "ADC.h"
#include <stdlib.h>	//for abs()
#include "SwTimer.h"

//#include "i2c.c"    ////****
//#include "HwtLowLevelDrivers.c" ////****

//temperature
#define TIMER_READ_TEMPERATURE_INTERVAL				5000	//ms

//height
#define TIMER_READ_HEIGHT_INTERVAL					500		//ms
#define TIMER_HEIGHT_WAITING_TO_RESTART_INTERVAL	500		//ms
#define TIMER_READ_HEIGHT_OFF_INTERVAL				500		//ms
#define NUMBER_OF_HEIGHT_READS						10
#define MAX_HEIGHT_READ_ATTEMPTS					10		//times

//sonar sensor
#define MAX_DIFFERENCE_BETWEEN_HEIGHT_READS			4		//cm
#define SENSOR_MAX_DISTANCE							170		//cm
#define	SENSOR_MIN_DISTANCE							20		//cm
#define DISTANCE_FROM_GROUND_TO_SONAR				217		//cm

//weight
#define TIMER_READ_WEIGHT_INTERVAL					250		//ms
#define NUMBER_OF_WEIGHT_READS						20




typedef enum
{
	EVENT_HWT_NONE,
	EVENT_HEIGHT_MEASUREMENT_ERROR,
	EVENT_HEIGHT_MEASUREMENT_COMPLETE_OK,
	EVENT_WEIGHT_MEASUREMENT_COMPLETE_OK,
	EVENT_TEMPERATURE_MEASUREMENT_COMPLETE_OK
} HeightEvents_e;

typedef enum
{
	HEIGHT_STATE_IDLE,
	HEIGHT_STATE_WAITING,
	HEIGHT_STATE_READING,
	HEIGHT_STATE_WAITING_TO_RESTART
} HeightStates_e;

static void HeightStateIdle(void);
static void HeigthStateWaiting(void);
static void HeigthStateReading(void);
static void HeigthStateWaitingToRestart(void);

const pFunc_t HeightStateMachine[] = 
{
	HeightStateIdle,
	HeigthStateWaiting,
	HeigthStateReading,
	HeigthStateWaitingToRestart
};


static void TimerReadHeight_Event(void);
static void TimerReadWeight_Event(void);
static void TimerReadTemperature_Event(void);


static UINT8 ActualState;

static UINT8 TimerReadHeight;
static UINT8 TimerReadWeight;
static UINT8 TimerReadTemperature;

static UINT8 HwtActualEvent = 0;
static UINT8 NumberOfReads = 0;

UINT8 Hwt_Height = 0;
UINT8 Hwt_Weight = 0;
UINT16 Hwt_Temperature = 0;





static void HeightStateIdle(void)
{
	//do nothing	
}

static void HeigthStateWaiting(void)
{
	//just waiting... aqu noms...	
}

static void HeigthStateWaitingToRestart(void)
{
	//just waiting... aqu noms...		
}

static void HeigthStateReading(void)
{
	static UINT8 actualHeight;
	static UINT8 lastHeight;	
	static UINT8 readAttempts = 0;
	
	lastHeight = actualHeight;			//store old hieght
	actualHeight = (UINT8)ReadSonar();	//read new height

	//Check if the reading is NOT in a valid range
	if ((actualHeight > SENSOR_MAX_DISTANCE) || (actualHeight < SENSOR_MIN_DISTANCE))
	{
		HWT_DISABLE_BOARD();
		
		//Error in the sensor, start again				
		if (readAttempts < MAX_HEIGHT_READ_ATTEMPTS)
		{
			readAttempts++;
			NumberOfReads = 0;
			ActualState = HEIGHT_STATE_WAITING_TO_RESTART;
			SwTimer_StartTimer(TimerReadHeight, TIMER_HEIGHT_WAITING_TO_RESTART_INTERVAL);	
		}
		else
		{
			//finish measurement
			//display error	
			readAttempts = 0;
			NumberOfReads = 0;
			ActualState = HEIGHT_STATE_IDLE;
			Hwt_Height = 0;
			HwtActualEvent = EVENT_HEIGHT_MEASUREMENT_ERROR;
		}				
	}
	else		//if measurement IS in a valid range
	{
		if (NumberOfReads < NUMBER_OF_HEIGHT_READS)
		{
			NumberOfReads++;
			SwTimer_StartTimer(TimerReadHeight, TIMER_READ_HEIGHT_INTERVAL);
			ActualState = HEIGHT_STATE_WAITING;
		}
		else
		{
			INT16 differenceBetweenHeights;
			
			NumberOfReads = 0;					
			differenceBetweenHeights = actualHeight - lastHeight;	
			
			//if the difference is in a valid range
			if (abs(differenceBetweenHeights) < MAX_DIFFERENCE_BETWEEN_HEIGHT_READS)
			{	
				HWT_DISABLE_BOARD();
				ActualState = HEIGHT_STATE_IDLE;
				readAttempts = 0;				
				Hwt_Height = DISTANCE_FROM_GROUND_TO_SONAR - actualHeight;
				HwtActualEvent = EVENT_HEIGHT_MEASUREMENT_COMPLETE_OK;
			}
			else	
			{
				//measurement was valid, but there was a large difference
				//between previous and actual reading
				//Start measurement again					
				
				if (readAttempts < MAX_HEIGHT_READ_ATTEMPTS)
				{
					readAttempts++;
					ActualState = HEIGHT_STATE_WAITING;
					SwTimer_StartTimer(TimerReadHeight, TIMER_READ_HEIGHT_INTERVAL);					
				}
				else
				{
					//finish measurement
					//display error	
					readAttempts = 0;
					ActualState = HEIGHT_STATE_IDLE;
					Hwt_Height = 0;
					HwtActualEvent = EVENT_HEIGHT_MEASUREMENT_ERROR;					
				}
			}
		}
	}	
}






void TimerReadHeight_Event(void)
{	
	if (ActualState == HEIGHT_STATE_WAITING)
	{		
		ActualState = HEIGHT_STATE_READING;	//go to next state
	}
	else if (ActualState == HEIGHT_STATE_WAITING_TO_RESTART)
	{		
		HWT_ENABLE_BOARD();
		ActualState = HEIGHT_STATE_WAITING;
		SwTimer_StartTimer(TimerReadHeight, TIMER_READ_HEIGHT_INTERVAL);	
	}
}


static void TimerReadWeight_Event(void)
{
	static UINT16 weightSum;
	
	if (NumberOfReads < NUMBER_OF_WEIGHT_READS)
	{
		UINT32 scaleResult = ReadScale();
		scaleResult = (UINT32)(scaleResult / (UINT32)1000);
		
		weightSum += (UINT16)(scaleResult);
		SwTimer_StartTimer(TimerReadWeight, TIMER_READ_WEIGHT_INTERVAL);
		NumberOfReads++;
	}
	else
	{
		weightSum = (UINT16)(weightSum/(UINT16)NUMBER_OF_WEIGHT_READS);	//calculate average of readings
		Hwt_Weight = (UINT8)(weightSum);		
		weightSum = 0;
		NumberOfReads = 0;
		HwtActualEvent = EVENT_WEIGHT_MEASUREMENT_COMPLETE_OK;	//trigger event
		HWT_DISABLE_BOARD();
	}
}



static void TimerReadTemperature_Event(void)
{
	Hwt_Temperature = ReadTemperatureSensor();
	Hwt_Temperature = (UINT16)(Hwt_Temperature / (UINT16)10);
	HwtActualEvent = EVENT_TEMPERATURE_MEASUREMENT_COMPLETE_OK;	//trigger event
	HWT_DISABLE_BOARD();
}


/* Public functions */

void Hwt_Init(void)
{
	HWT_BOARD_ENABLE_PIN_DD = 1;	
	HWT_ENABLE_BOARD();	

	ADC_Init12b(1<<ADC_WEIGHT_CHANNEL);
	W_tare();	//need to call this several times to allow the
	W_tare();
	W_tare();
	W_tare();
	(void)ReadTemperatureSensor();		//dummy read
	
	//create timers
	TimerReadHeight = SwTimer_CreateTimer(TimerReadHeight_Event);
	TimerReadWeight = SwTimer_CreateTimer(TimerReadWeight_Event);;
	TimerReadTemperature  = SwTimer_CreateTimer(TimerReadTemperature_Event);
	
	HWT_DISABLE_BOARD();
}


void Hwt_StartReadingTemperature(void)
{
	HWT_ENABLE_BOARD();
	SwTimer_StartTimer(TimerReadTemperature, TIMER_READ_TEMPERATURE_INTERVAL);
}

void Hwt_StartReadingWeight(void)
{
	HWT_ENABLE_BOARD();
	ADC_Init12b(1<<ADC_WEIGHT_CHANNEL);
	SwTimer_StartTimer(TimerReadWeight, TIMER_READ_WEIGHT_INTERVAL);
}



void Hwt_StartReadingHeight(void)
{
	if (ActualState == HEIGHT_STATE_IDLE)
	{	
		HWT_ENABLE_BOARD();
		ActualState = HEIGHT_STATE_WAITING;
		SwTimer_StartTimer(TimerReadHeight, TIMER_READ_HEIGHT_INTERVAL);	
	}
	else
	{

	}
}


void Hwt_StopReadingTemperature(void)
{
	SwTimer_StopTimer(TimerReadTemperature);
	HWT_DISABLE_BOARD();	
}


void Hwt_StopReadingHeight(void)
{
	SwTimer_StopTimer(TimerReadHeight);	
	NumberOfReads = 0;
	HWT_DISABLE_BOARD();	
}

void Hwt_StopReadingWeight(void)
{
	SwTimer_StopTimer(TimerReadWeight);	
	NumberOfReads = 0;
	HWT_DISABLE_BOARD();
}


void Hwt_PeriodicTask(void)
{
	//execute state machine	
	HeightStateMachine[ActualState]();
	
	/* Event handler */
	if (HwtActualEvent != EVENT_HWT_NONE)
	{
		if (Hwt_Events[HwtActualEvent] != NULL)
		{
			Hwt_Events[HwtActualEvent]();	//execute registered event
			HwtActualEvent = EVENT_HWT_NONE;
		}
	}	
}
